#/usr/bin/python3
#coding: utf-8


"""
Vérification grammaticale des candidats.

Etapes principales : 
- étiquetage POS des candidats
- comparaison avec la phrase modèle
- parsing en constituants des candidats
- vérification de la réalisation des phénomènes d'accord

"""

from generation import *
# from parser import *
# from parser2 import *
# from grivoiserie import *
import subprocess
import json

with open('reec_stanford.txt', 'r') as file:
    reec_stanford = json.load(file)

# équivalences pour la simplification des POS tags
equiv_tags = {
	"v" : ["VERB","VINF","V","VPP"],
	"nc" : ["NC","NPP","ADJ"],
	"np" : ["NC","NPP"],
	"adj" : ["ADJ","N"],
	"prep" : ["P","DET"],
	"det" : ["DET","P"],
	"adv" : ["ADV"],
	"ponctw" : ["PUNC"],
}

# équivalences entre les POS tags de l'analyseur de Stanford et celles du lexique
equiv_stmlex = {
	"DET" : "det",
	"NC" : ["nc","np"],
	"NP" : ["nc","np"],
	"P" : ["det","prep"],
	"DET" : ["det","p"], # là
	"ADJ" : "adj",
	"N" : ["nc","np"],
	"NPP" : ["nc","np"],
	"PRO" : "cln",
	"CLO" : ["cla","cld"],
	"V" : "v",
	"CS" : "que",
	"CLS" : "cln",
	"VPP" : "v",
	"VINF" : "v",
	"CLR" : ["clr","cldr","clar"],
	"VPR" : "v",
	"ADV" : "adv",
	"PROREL" : "pro",
	"PROWH" : "pri",
	"C" : "conj"
}


def check_reec(phrase,regles):
	"""
		Vérifier la conformité de la structure d'une phrase avec des règles de réécritures.
		- prend en entrée : la phrase et l'ensemble des règles
		- renvoit en sortie : 'True' si la réécriture est correcte, 'False' sinon

	"""
	phrase_parsee = stanford_lexparse(phrase)
	pp = re.sub("\n","",phrase_parsee)
	pp = re.sub("\s+"," ",pp)
	synt_pp = list(parenthetic_contents(pp))
	for synt in synt_pp:
		if is_unitaire( "("+synt[1]+")") == True:
			continue
		else:
			c_parent = synt[1].split()[0]
			# print("\n",synt[1])
			deps = [dep[1] for dep in list(parenthetic_contents(synt[1])) if dep[0]== 0]
			c_enfants = [d.split()[0] for d in deps]
			if c_enfants in regles[c_parent]:
				return True
			else:
				return False

def get_mlex(file,enc):
	"""
		Charger depuis un fichier le lexique mlex au format dictionnaire de python.
		- prend en entrée : le nom du fichier et son encodage
		- renvoie en sortie : le dictionnaire python
	"""
	mlex = defaultdict(list)
	with codecs.open(file, "r",encoding=enc) as mlex_file:
		for line in mlex_file:
			line = line.strip()
			data = line.split("\t")
			if len(data) ==3:
				data.append("")
			mlex[data[0]].append([data[1],data[3]])
	return mlex

# ======================================================

### Chemins des fichiers (à remplacer éventuellement)

fichier_LIUM = "lexique_lium_utf8.txt"
fichier_MLEX = "lefff-3.4.mlex"
fichier_LIAPHON = "contrepeteries_utf8.phono"
fichier_ELEX_MINI = "mini.elex"
fichier_ELEX = "lefff-3.4_utf8.elex"

# ======================================================

### Chargement des lexiques : 

# lexiques LIUM : 
lexique_o2p = extraire_lexique(fichier_LIUM,sens="o2p")
lexique_p2o = extraire_lexique(fichier_LIUM,sens="p2o")

# lexique mlex (lefff-3.4.mlex)
mlex = get_mlex(fichier_MLEX,"utf-8")

# ======================================================



def get_cat(token,equivalences):
	"""
	Obtenir les catégories possibles pour un token. 
	Fonctionne avec les équivalences de simplfication (voir plus haut dans ce script)
	- prend en entrée : le token et le dictionnaire d'équivalences
	- renvoit en sortie : les catégories grammaticales possibles

	"""
	cats = []
	entries = mlex[token]
	for entry in entries:
		cat = entry[0]
		cats.append(equivalences[cat])
	return cats

def stanford_pos_tag(sentence):
	"""
	Etiqueter en POS une phrase avec l'étiqueteur de Stanford.
	- prend en entrée : la phrase (/le texte) à étiqueter.
	- renvoit en sortie : la phrase (/le texte) étiquetée.

	"""
	sentence = bytes(sentence, "utf-8")
	p = subprocess.Popen(["java","-mx300m","-cp",
	"/home/gu/Téléchargements/stanford-postagger-full-2017-06-09/stanford-postagger.jar:","edu.stanford.nlp.tagger.maxent.MaxentTagger",
	"-model","/home/gu/Téléchargements/stanford-postagger-full-2017-06-09/models/french.tagger"],
		stdin=subprocess.PIPE,stdout=subprocess.PIPE)
	parse_stdout = p.communicate(input=sentence)[0]
	return parse_stdout.decode("utf-8").strip()

def stanford_lexparse(string):
	"""
	Analyser en constituants une phrase (/ un texte) avec l'analyseur de Stanford.
	- prend en entrée : la phrase (/le texte)
	- renvoit en sortie : le texte étiqueté en constituants

	"""
	string = bytes(string, "utf-8")
	# sh ./lexparser.sh data/testsent.txt
	p = subprocess.Popen(["java","-Xmx2g","-cp","/home/gu/Téléchargements/stanford-parser-full-2017-06-09/*:",
		"edu.stanford.nlp.parser.lexparser.LexicalizedParser",
		"/home/gu/Téléchargements/stanford-french-corenlp-2017-06-09-models/edu/stanford/nlp/models/lexparser/frenchFactored.ser.gz","-"],

		stdin=subprocess.PIPE,stdout=subprocess.PIPE)
	parse_stdout = p.communicate(input=string)[0]
	return parse_stdout.decode("utf-8").strip()

def stanford_nnparse(string):
	""" 
	Parser des chaînes de caractères avec le parseur réseau de neurones de Stanford.
	 - prend en entrée : une string (ex. "Je suis une phrase de test. Moi aussi.") 
	 - renvoit en sortie : une string contenant les relations de dépendance pour cette phrase : 
		. chaque relation se termine par '\n'
		. les relations pour les phrases de l'input sont séparées par des lignes vides.

	"""
	string = bytes(string, "utf-8")
	p = subprocess.Popen(["java","-Xmx2g","-cp","/home/gu/Téléchargements/stanford-parser-full-2017-06-09/*:",
		"edu.stanford.nlp.parser.nndep.DependencyParser",
		"-model",
		"/home/gu/Téléchargements/stanford-french-corenlp-2017-06-09-models/edu/stanford/nlp/models/parser/nndep/UD_French.gz",
		"-tagger.model",
		"/home/gu/Téléchargements/stanford-french-corenlp-2017-06-09-models/edu/stanford/nlp/models/pos-tagger/french/french-ud.tagger",
		"-textFile","-"],

		stdin=subprocess.PIPE,stdout=subprocess.PIPE)
	parse_stdout = p.communicate(input=string)[0]
	return parse_stdout.decode("utf-8").strip()


def split_tokens_tags(parsed_sentence):
	"""
		Séparer les tokens et leurs étiquettes dans la sortie de l'analyseur de Stanford.
		- prend en entrée : la phrase parsée
		- renvoit en sortie : la liste des tokens et la listes des étiquettes (constituants)

	"""
	toks, tags = [],[]
	tokens = parsed_sentence.split()
	for token in tokens:
		t,tag = token.split("_")
		toks.append(t)
		tags.append(tag)
	return toks,tags

def simplify_taglist(taglist):
	"""
		Réduire un tagset.
		- prend en entrée : une liste d'étiquettes (tags)
		- renvoit en sortie : une liste simplifiée

	"""
	new_taglist = []
	for tag in taglist:
		if tag in ["NPP","NC"]:
			tag = "N"
		elif tag in ["ADJ"]:
			tag = "N"
		elif tag in ["VIMP"]:
			tag = "V"
		new_taglist.append(tag)
	return new_taglist


def test(phrase,patron):

	contrepets = []

	c = Contrepet(*separer_ortho_phono(lia_phonetisation(phrase)))
	# c_tokens,c_tags = split_tokens_tags(stanford_pos_tag(phrase))
	# print("c tags",c_tags)
	# print("c tokens",c_tokens)
	# print(c.phono)
	c.underscores(lexique_o2p)

	print("\nApplication du patron de permutations...")
	test = c.appliquer_patron(lexique_o2p,patron)

	print("Ce patron permet de générer {} candidat(s) phono.".format(len(test)/2))
	
	if  test is None or len(test) == 0:

		return None

	else:
		# print(test)
		c_tokens,c_tags = split_tokens_tags(stanford_pos_tag(phrase))
		candidats_txt = []

		
		print("\nGénération des orthographes possibles...")
		check = defaultdict() # vérifiés corrects
		wrong = []
		wrong_strings = []
		for t in test: # pour chaque candidat contrepèterie
			# print(t)
			if c.to_ortho(t,lexique_p2o) is not None: # si on peut l'écrire
				for o in c.to_ortho(t,lexique_p2o): # pour chaque orthographe de ce candidat
					print("o",o)
					o = " ".join(o)+"."
					# o = o[0].upper() + o[1:]
					candidats_txt.append(o)
		candidats_txt = list(set(candidats_txt))
		candidats_txt = " ".join(candidats_txt)
		# print("les candidats",candidats_txt)
		if len(candidats_txt) == 0:
			return None

		else:
			# print("les candidats",candidats_txt)
			print("\n Première vérification syntaxique des candidats...")
			parsed_text = stanford_pos_tag(candidats_txt)
			parsed_candidats = parsed_text.split("\n")
			for pc in parsed_candidats: # pour chaque candidat parsé
				pc_tokens, pc_tags = split_tokens_tags(pc) # ses tokens, les pos tags de ses tokens
				# print("\nnouveau candidat : ",pc_tokens)
				# print("\n",pc_tags,c_tags)
				# print("\n",simplify_taglist(pc_tags), simplify_taglist(c_tags))
				if simplify_taglist(pc_tags) == simplify_taglist(c_tags): # si les pos tags du candidat sont les mêmes que celles de l'input
					# print("\n",simplify_taglist(pc_tags), simplify_taglist(c_tags))
					nb_tokens_diff = 0
					if len(pc_tags) == len(c_tags):
						for i in range(len(pc_tokens)-1): 
							if pc_tokens[i].lower() != c.ortho[i].lower(): # pour chaque token du candidat différent de celui en input au même index
								nb_tokens_diff += 1
						# print("le nombre de tokens différents",nb_tokens_diff)

					nb_diffs_ok = 0
					for i in range(len(pc_tokens)-1):
						if pc_tokens[i].lower() != c.ortho[i].lower(): 
							score_equiv = 0
							# print(pc_tokens[i],pc_tags[i])
							equiv_cat_pc_token = get_cat(pc_tokens[i],equiv_tags)
							# print(pc_tokens[i],equiv_cat_pc_token,"contient ?",c_tags[i])
							for equiv_cat in equiv_cat_pc_token:
								for eq_c in equiv_cat:
									if c_tags[i] in equiv_cat:
										score_equiv += 1
							if score_equiv > 0:
								nb_diffs_ok += 1
					# print("Le nombre de tokens différents avec la même POS",nb_diffs_ok)
					if nb_tokens_diff == nb_diffs_ok:
						# print("OUIIIIII")
						contrepets.append(" ".join(pc_tokens[:-1])+".")
			if len(contrepets) > 0:
				return contrepets
			else:
				return None


def separer_phrases(output):
	"""
		Séparer les phrases dans l'output du parseur de Stanford.
		- prend en entrée : l'output du parser
		- renvoit en sortie : une liste de phrases parsées.
	"""
	sents, tmp = [],[]
	output = output.split("\n")
	for rel in output[:-1]:
		if rel != "":
			# print(rel,"yy")
			tmp.append(rel)
		else:
			# print("---")
			sents.append(tmp)
			tmp = []
	tmp.append(output[-1])
	sents.append(tmp)
	return sents

def decomposer_relation(relation):
	""" 
		Récupérer les différents éléments de la relation de dépendance.
		- en entrée : une relation (ex. : "nsubj(arrivé-3, Je-1)")
		- en sortie : un tuple de trois éléments
			. rel = le type de relation
			. head = la tête
			. dep = le dépendant

	"""
	pat = "(?P<rel>\w+)\((?P<head>[a-zA-Zéèêùûîïàâô]+)\-\d+,\s(?P<dep>[\.a-zA-Zéèêùûîïàâô]+)\-\d+\)"
	m = re.match(pat,relation)
	if m:
		return m.group("rel"), m.group("head"), m.group("dep")
	else:
		return "Souci dans la regex ! "


def parenthetic_contents(string):
	"""
	Découper une chaîne en fonction du niveau d'enchâssement dans les parenthèses.
	Ex. de chaîne à traiter : '(NP (D le) (N chat))'
	- prend en entrée : la chaîne à découper
	- renvoit en sortie : le découpage au fur et à mesure (transformer cet
	output en list pour qu'il soit exploitable)

	"""
	stack = []
	for i, c in enumerate(string):
		if c == '(':
			stack.append(i)
		elif c == ')' and stack:
			start = stack.pop()
			yield (len(stack), string[start + 1: i])


def extraire_traits(token,cat,lexique):
	"""
		Récupérer les traits morphologiques d'un token dans un lexique.
		- prend en entrée : le token, sa catégorie grammaticale, et le lexique mlex
		- renvoit en sortie : les traits possibles pour la graphie du token (transformer 
		cet	output en list pour qu'il soit exploitable)

	"""
	if token not in lexique:
		# print("c'est là que ça merde")
		return None
	else:
		data = lexique[token]
		# data = list(set(data))
		for poss in data:
			if poss[0] in equiv_stmlex[cat]:
				# print("oui")
			# if cat in poss[0]:
				s = {}
			# genre
				if "m" in poss[1]:
					s["genre"] = "m"
				elif "f" in poss[1]:
					s["genre"] = "f"
			# nombre
				if "s" in poss[1]:
					s["nombre"] = "s"
				elif "p" in poss[1]:
					s["nombre"] = "p"
			# personne
				m = re.findall("\d",poss[1])
				if m:
					if len(m) == 1:
						s["pers"] = m[0]
						yield s
					else:
						for p in m:
							s2 = dict(s)
							s2["pers"] = p
							yield s2
				else:
					yield s

def uniq_traits(token,cat,lexique):
	"""
		Supprimer les doublons dans la liste des traits morpho. possibles pour un token.
		- prend en entrée le token, sa catégorie grammaticale et le lexique de traits.
		- renvoit en sortie : la liste des traits possibles, sans les doublons.

	"""
	toutes_poss = extraire_traits(token,cat,lexique)
	# [print(poss) for poss in toutes_poss]
	uniq = []
	for poss in toutes_poss:
		if poss not in uniq:
			uniq.append(poss)
	return uniq

def comparer_traits(poss,cigle):
	"""
		Vérifier l'unification des traits dans des phénomènes d'accords.
		- prend en entrée : les structures de traits à comparer et le cigle du syntagme 
			à l'intérieur duquel on vérifie le phénomène d'accord
		- renvoit en sortie : la structure de traits du syntagme le cas échéant, 'None' en
			cas de violation de l'accord.

	"""
	poss2 = [s for s in poss if len(s.keys()) > 0]
	if cigle == "PP":
		# tmp_nombre = [s["nombre"] for s in poss2 if "nombre" in s.keys()]
		# if len(tmp_nombre) == 1:
		# 	return [{}]
		# elif tmp_nombre[0] == "p" and tmp_nombre[1] == "s":
		# 	return None
		# else:
		return [{}]
	elif cigle in ["NP","MWN"]:
		tmp_nombre = [s["nombre"] for s in poss2 if "nombre" in s.keys()]
		tmp_genre = [s["genre"] for s in poss2 if "genre" in s.keys()]
		tmp_nombre = list(set(tmp_nombre))
		tmp_genre = list(set(tmp_genre))
		if len(tmp_nombre)>1 or len(tmp_genre)>1:
			return None
		else:
			return [{"nombre" : tmp_nombre[0] , "genre":tmp_genre[0]}]
	elif cigle == "VN":
		tmp_nombre = [s["nombre"] for s in poss2 if "nombre" in s.keys()]
		tmp_pers = [s["pers"] for s in poss2 if "pers" in s.keys()]
		tmp_nombre = list(set(tmp_nombre))
		tmp_pers = list(set(tmp_pers))
		if len(tmp_pers) > 1 or len(tmp_nombre) > 1:
			return None
		else:
			return [{"nombre" : tmp_nombre[0], "pers":tmp_pers[0]}]







def is_unitaire(syntagme):
	"""
		Vérifier si un syntagme est unitaire (format de l'output de Stanford).
		- prend en entrée : le syntagme au format mentionné
		- renvoit en sortie : 'True' si le syntagme est unitaire, 'False' sinon

	"""
	pat = "\([A-Z]+([a-z]+)?\s\w+\)$"
	m = re.match(pat,syntagme)
	if m:
		return True
	else:
		return False


def verif(pc,connus,wrong):
	"""
		Vérifier si un syntagme a déjà été analysé, et s'il est grammaticalement incorrect.
		- prend en entrée : le syntagme, un dictionnaire de syntagme examinés, et un
			dictionnaire de syntagmes erronnés
		- renvoit en sortie : "OK" si le syntagme est correct (connu ou non)
							  "NOK" si le syntagme est erroné (connu ou non)
		NB : enrichissement des dictionnaires au fur et à mesure de l'analyse de nouveaux
		syntagmes
	"""
		pc = " ".join(pc)
		# pc = re.sub("\s+"," ",pc)
		# print("\n",pc)
		syntagmes = parenthetic_contents(pc)
		d = defaultdict(list)
		for s in syntagmes:
			d[s[0]].append(s[1])
		levels = list(d.keys())
		levels = sorted(levels,reverse=True)
		for l in levels:
			a_verif = d[l]
			for x in a_verif:
				# print("\n",x)
				if is_unitaire("("+x+")") == True:
					cat,token = x.split()
					traits = uniq_traits(token,cat,mlex)
					# print(token,traits)
					if traits is not None:
						connus["("+x+")"] = traits
					else:
						return "NOK"
				else:
					# print(x)
					cigle = x.split()[0]
					# print("cigle",cigle)
					if cigle not in ["ROOT","SENT","PUNC","Ssub","Srel","VPinf","VPpart","Sint"]:
						deps = [dep[1] for dep in list(parenthetic_contents(x)) if dep[0]== 0 and "CLO" not in dep[1]]
						if len(deps) == 1:
							# print(x,connus["("+deps[0]+")"])
							connus["("+x+")"] = connus["("+deps[0]+")"]
						else:
							# print(x)
							tous = []
							for dep in deps:
								# print("dep",dep)
								if "("+dep+")" in connus and dep.split()[0] not in ["CLO","VPP","VPR","ADV"]:
									# print("on le connait",dep,connus["("+dep+")"])
									tous.append(connus["("+dep+")"])
							# print("tous",tous)
							toutes_poss = list(itertools.product(*tous))
							nb_ok = 0
							for poss in toutes_poss:
								# print(poss)
								# print("on compare",poss,"-->",comparer_traits(poss,cigle))
								if comparer_traits(poss,cigle) is not None:
									connus["("+x+")"] = comparer_traits(poss,cigle)
									nb_ok +=1
								else:
									wrong.append(x)
							if nb_ok == 0:
								return "NOK"
		return "OK"


def contrepeterie(phrase,patron):
	"""
		Conserver les candidats pour lesquels aucune violation n'a été relevée.
		- prend en entrée : la phrase (string) et un patron
		- renvoit en sortie : une liste de candidats 
	"""
	ok = []
	res3 = test(phrase,patron)

	if res3 is None:
		print("\nIl n'y a pas de contrepèterie possible pour cette phrase avec ce patron.")
		return None

	else:
		wrong = []
		connus = defaultdict()
		tout_texte = "\n".join(res3)

		print("\n Seconde vérification des candidats...")
		parsed_texte = stanford_lexparse(tout_texte)
		parsed_candidats = separer_phrases(parsed_texte)

		for pc in parsed_candidats:
			index = parsed_candidats.index(pc)
			print(res3[index])
			if verif(pc,connus,wrong) == "OK":
				# print(res3[index])
				ok.append(res3[index])
	return ok