#/usr/bin/python3
#coding: utf-8


"""
Génération des candidats à la contrepèterie. 

Etapes principales : 
- phonétisation
- segmentation en syllabes et structuration des syllabes
- application des patrons de permutation : 
	- matching des éléments de permutation
	- application des échanges
- retranscription orthographique des candidats quand c'est possible

"""

import argparse, codecs, sys, re
from collections import defaultdict
import treetaggerwrapper
import itertools
import subprocess


tagger = treetaggerwrapper.TreeTagger(TAGLANG="fr")

def lia_phonetisation(sentence):
	""" Appel de l'outil de phonétisation du LIA.
		- prend en entrée : une phrase (transcription orthographique)
		- renvoit en sortie : la transcription phonétique de cette phrase 
			NB : les transcriptions des mots (ortho + phono) sont séparés par des tabulations ; 
			un phonème = 2 caractères ASCII (voir annexes du mémoire pour plus d'informations) 
	"""
	LIA_PHON_REP = "/media/sf_memoire/lia_phon/script/" # remplacer par le chemin du répertoire
	sentence = bytes(sentence, "iso-8859-1")
	p = subprocess.Popen(["perl", LIA_PHON_REP +"lia_text2phon"],
		stdin=subprocess.PIPE,stdout=subprocess.PIPE)
	parse_stdout = p.communicate(input=sentence)[0]
	return parse_stdout.decode("iso-8859-1").strip()


def separer_ortho_phono(lia_output):
	"""
	Séparer les transcriptions phonétiques et orthographiques des sorties de l'outil LIA PHON.
		- prend en entrée : une sortie de l'outil LIA PHON
		- renvoit en sortie : une liste de deux éléments (1.transcription ortho ; 2. transcription phono)	
	"""
	lia_output = lia_output.split("\n")
	contrepeteries_phono = []
	contrepeteries_ortho = []
	cp = []
	co = []
	pat = re.compile("(['\w]+)\s+([\?\w]+)\s+\[")
	for line in lia_output:
		line = line.strip()
		match = re.search(pat,line)
		if match:
			ortho, phono = match.groups()
			cp.append(phono)
			co.append(ortho)
		else:
			if line == "</s> ## [ZTRM->EXCEPTION]":
				contrepeteries_phono.append(cp)
				contrepeteries_ortho.append(co)
				cp = []
				co = []
			elif line in ["<s> ## [ZTRM->EXCEPTION]","pause ##"]:
				continue
	return contrepeteries_ortho[0], contrepeteries_phono[0]

def extraire_lexique(fichier,sens="o2p"):
	"""
	"""
	print("\nChargement du lexique phonétique du LIUM depuis {} (sens = {}) ...".format(fichier,sens))
	lium_lex_o2p = defaultdict(lambda: defaultdict(str))
	lium_lex_p2o = defaultdict(list)
	# pat = re.compile("-?(\w+)\((\d?)\)\s(.*)$")
	pat = re.compile("-?(?P<ortho>[\w-]+)s?\(?(?P<num>\d)?\)?\s?(?P<phono>.*)$")
	with codecs.open(fichier, "r", encoding="utf-8") as lium_file:
		for line in lium_file:
			m = re.search(pat,line)
			if m:
				# print("GROUPS : ",m.groups())
				data = list(m.groups())
				if data[1] == None:
					data[1] = 1
				if len(data) == 3:
					data[1] = int(data[1])-1
				data[2] = re.sub("\s","",data[2])
				if sens == "o2p":
					lium_lex_o2p[data[0]][data[1]] = data[2]	# 0 = ortho ; 1 = numéro (fac.) ; 2 = transcription phono
				elif sens == "p2o":
					lium_lex_p2o[data[2]].append(data[0])			
	if sens == "o2p":
		print("Le lexique {} contient {} entrées.".format(sens,len(lium_lex_o2p)))
		return lium_lex_o2p
	elif sens == "p2o":
		print("Le lexique {} contient {} entrées.".format(sens,len(lium_lex_p2o)))
		return lium_lex_p2o

def combine(variantes):
	tmp = []
	nums = []
	for orthos in variantes:
		tmp.append(orthos[:-1])
		n = orthos[-1]
		if n not in nums:
			nums.append(n)
	poss = list(itertools.product(*tmp))
	return [nums,poss]

def phonosplit(mot):
	phones = []
	for n in range(0,len(mot),2):
	    m = n+2
	    phones.append(mot[n:m])
	return phones



def extraire_contrepeteries(f):
	"""
		Extraire les contrepeteries du fichier de transcriptions phonémiques.
		- Prend en entrée : (fichier) liste de transcriptions phonémiques, une contrepeterie par ligne.
		- Renvoie en sortie : une liste
	"""
	contrepeteries_phono = []
	contrepeteries_ortho = []
	with codecs.open(f,"r",encoding="utf-8") as inputFile:
		cp = []
		co = []
		pat = re.compile("(['\w]+)\s+([\?\w]+)\s+\[")
		for line in inputFile:
			line = line.strip()
			match = re.search(pat,line)
			if match:
				ortho, phono = match.groups()
				cp.append(phono)
				co.append(ortho)
			else:
				if line == "</s> ## [ZTRM->EXCEPTION]":
					contrepeteries_phono.append(cp)
					contrepeteries_ortho.append(co)
					cp = []
					co = []
				elif line in ["<s> ## [ZTRM->EXCEPTION]","pause ##"]:
					continue
	return contrepeteries_ortho, contrepeteries_phono

class Element:
	def __init__(self,syllabe,nature,nb_syllabes):
		self.syllabe = syllabe
		self.nature = nature
		self.nb_syllabes = nb_syllabes


# définition des éléments de permutation: 
e0 = Element(2,"AC",3)
e1 = Element(1,"AC",1)
e2 = Element(1,"AC",2)
e3 = Element(2,"AC",2)
e4 = Element(1,"ACC",1)
e5 = Element(1,"ACC",2)
e6 = Element(2,"AC+N",2)
e7 = Element(1,"AC+N",2)
e8 = Element(1,"AC1",1)	
e9 = Element(1,"AC1",2)
e10 = Element(2,"AC1",2)
e11 = Element(1,"AC2",1)
e12 = Element(1,"AC2",2)
e13 = Element(1,"AC2+N",1)
e14 = Element(1,"AC2+N",2)

# définition des permutations
permut0 = [e1,e1]
permut1 = [e1,e2]
permut2 = [e5,e2]
permut3 = [e2,e2]
permut4 = [e1,e11]
permut5 = [e2,e3]
permut6 = [e8,e1]
permut7 = [e4,e1]
permut8 = [e5,e1]
permut9 = [e0,e1]
permut10 = [e3,e1]
permut11 = [e2,e8]
permut12 = [e4,e3]
permut13 = [e3,e1]
permut14 = [e2,e2]
permut15 = [e2,e4]
p = [e0,e2]

class Contrepet:
	"""
		Classe principale du prorotype, modélisant l'ensemble des états intermédiaires
		de la phrase et de son contrepet. Contient les fonctions de découpage et de transformation
		de ces états intermédiaires. 

	"""
	def __init__(self, ortho, phono):
		self.ortho = ortho
		self.phono = phono

	def underscores(self,lexique):
		"""
			Reconstruire les transcriptions orthographiques et phonétiques des items
			dans latrascription desquels l'outil LIA PHON incorpore des underscores.
			- prend en entrée : un objet Contrepet + un lexique de transcriptions phonétiques
			- renvoie en sortie : les transcriptions orthographiques et phonétiques corrigées.

		"""
		for token in self.ortho:
			if "_" in token:
				n = self.ortho.index(token)

				# découpage de la liste en deux
				# ortho
				subso0 = self.ortho[:n]
				subso1 = self.ortho[n+1:]

				#phono
				subsp0 = self.phono[:n]
				subsp1 = self.phono[n+1:]

				subtokens = token.split("_")

				for st in subtokens :
					
					subphon = lexique[st][0]

					subso0.append(st)
					subsp0.append(subphon)

				self.ortho = subso0 + subso1
				self.phono = subsp0 + subsp1
			else:
				continue
		return self.ortho, self.phono



	def mots_pleins(self,lexique):
		"""
			Déterminer quels mots, selon leur étiquette POS, sont des mots pleins.
			- prend en entrée : un objet Contrepet
			- renvoit en sortie : une liste indiquant pour chaque mot du Contrepet si
				c'est un mot plein (1) ou non (0).
				ex : Le petit chat <-> [O,1,1]
		"""
		ortho = " ".join(self.underscores(lexique)[0])
		ortho = re.sub("[\.\?!;,]","",ortho)	
		ortho = re.sub("' ","'",ortho)
		treetagged_words = tagger.tag_text(ortho)
		pleins = []
		for w in treetagged_words:
			if w.split("\t")[1] in ["ADJ","ADV","NAM","NOM","SYM","VER:cond conditional",
			"VER:futu futur","VER:impe imperative","VER:impf","VER:infi","VER:pper",
			"VER:ppre","VER:pres","VER:simp","VER:subi","VER:subp"]:
				pleins.append(1)
			else:
				pleins.append(0)
		return pleins


	def syllabation(phones):
		"""
			- Prend en entrée : un token sous forme de liste de phones
			- Renvoie en sortie : une liste de phones segmentée en syllabes

			Règles à appliquer (Léon, Bhatt, Baligand - 1992) :
				- Si un phone est une voyelle, c'est un noyau (R1);
				- Si deux consonnes se suivent et que C1 est une occlusive et C2 une liquide, elles forment un cluster (R2);
				- Si deux consonnes se suivent sans former un cluster, alors C1 appartient à la syllabe de gauche et 
				  C2 à la syllabe de droite (R3);
				- Si une consonne est entre deux voyelles, elle appartient à la syllabe de droite (= elle
				  forme l'attaque consonnantique de la syllabe dont la voyelle de droite est le noyau) (R4);
		"""

		noyaux = ["ii","ei","ai","aa","oo","au","ou","uu","EU","oe","eu","an","un","on","ee","in"]
		occlusives_fricatives = ["tt","pp","gg","kk","mm","bb","nn","ff","vv","dd","jj","ss","ch"]
		liquides_semiv = ["rr","ll","ww","yy"]

		# segmentation tous les deux caractères pour identification des transcriptions de phones
		tmp_phones = []
		for i in range(len(phones)+1):
			tmp_phones.append(phones[i:i+2])
		phones = []
		[phones.append(x) for x in tmp_phones[0:-1:2]]
		# print("phones",phones)
		# identifier les noyaux et les clusters du token
		# tmp = liste de tuples(phone,Noyau / Consonne)
		tmp = []
		for i in range(len(phones)-1):

			#chercher les noyaux (R1)
			if phones[i] in noyaux:
				tmp.append((phones[i],"N"))
			#chercher les clusters et les regrouper en une seule unité (R2)
			elif phones[i] in occlusives_fricatives and phones[i+1] in liquides_semiv:
				tmp.append((phones[i]+phones[i+1],"Cl"))
			elif phones[i] in occlusives_fricatives and phones[i+1] in occlusives_fricatives and i+1 == len(phones)-1:
				tmp.append((phones[i]+phones[i+1],"Cl"))
			#identifier le reste (qui doivent être des consonnes / des semi-v)
			else:
				tmp.append((phones[i],"C"))
		if phones[-1] in noyaux:
			tmp.append((phones[-1],"N"))
		else:
			tmp.append((phones[-1],"C"))

		# suppression des C excédentaires qui sont des C2 de leur cluster
		for i in range(len(phones)-2):
			if tmp[i][1] == "Cl" and tmp[i+1][1] == "C" and tmp[i][0][2:] == tmp[i+1][0]:
				# print(tmp[i],tmp[i+1])
				tmp.remove(tmp[i+1])
				# break


		# reconstruire les syllabes du token 
		syllabes = []

		# application de R3 
		for i in range(len(tmp)-1):
			if tmp[i][1] == "C" and tmp[i+1][1] in ["C","Cl"]:
				tmp.insert(i+1,("-","S"))

		# application de R4
		s_positions = []
		# print("tmp",tmp)
		for i in range(len(tmp)-2):
			if tmp[i][1] == "N" and tmp[i+1][1] in ["C","Cl"] and tmp[i+2][1] == "N":
				s_positions.append(int(i+1))
		j,k = 0,0
		while j < len(s_positions):
			tmp.insert(s_positions[j]+k,("-","S"))
			j += 1
			k += 1
		# reconstruction des syllabes
		s = []
		for i in range(len(tmp)):
			if tmp[i][0] != "-":
				s.append(tmp[i])
			else:
				syllabes.append(s)
				s = []
		if len(s)>0:
			syllabes.append(s)
		return syllabes


	def structure_syllabe(syllabe):
		"""
			Structurer la syllabe : attaque / noyau / coda
			- Prend en entrée une syllabe sous la forme d'une liste de phones
			- Renvoie la syllabe structurée sous forme de dictionnaire

		"""
		a, n, c = [], [], []

		# si la syllabe commence par un noyau (= pas d'attaque consonantique)
		if syllabe[0][1] == "N":
			n.append(syllabe[0][0])
			# toutes les consonnes qui suivent sont néecessairement dans la coda
			for p in syllabe[1:]:
				c.append(p[0])

		# si la syllabe commence par un cluster consonantique
		elif syllabe[0][1] == "Cl":
			a.append(syllabe[0][0])
			# il ne peut pas y avoir de consonnes en plus du cluster dans l'attaque
			for p in syllabe[1:]:
				if p[1] == "N":
					n.append(p[0])
				elif p[1] == "C" or p[1] == "Cl":
					c.append(p[0])

		# si la syllabe commence par une consonne qui ne fait pas partie d'un cluster			
		elif syllabe[0][1] == "C":
			a.append(syllabe[0][0])
			if len(syllabe) > 1:
				# le second phone est forcément un noyau
				n.append(syllabe[1][0])
				# il n'y a qu'un noyau, le reste est donc consonantique et dans la coda
				for p in syllabe[2:]:
					c.append(p[0])
		structure = {"a":a, "n":n, "c":c}
		return structure



	def get_syllabes(self,lexique):
		"""
			Segmentation en syllabes de la transcription phonémique d'une contrepeterie
			- Prend en entrée : une contrepeterie transcrite en phones
			- Renvoie en sortie : une liste de structures syllabiques (dictionnaires attaque / noyau / coda)
		"""
		contrephono = []

		# pour chaque 'mot'
		for token in self.underscores(lexique)[1]:

			phono = Contrepet.syllabation(token)
			# print("\n",phono)
			syllabes = []

			# pour chaque syllabe du mot
			for syllabe in phono : 
				structure = Contrepet.structure_syllabe(syllabe)
				# print("structure",structure)
				syllabes.append(structure)

			contrephono.append(syllabes)
		return contrephono



	def match(self,element,lexique):
		"""
			Obtenir l'ensemble des positions dans une phrase auxquelles un élément est repéré.
			- prend en entrée : un objet Contrepet, un élément, et un lexique (ortho -> phono)
			- renvoit en sortie : la liste des positions auxquelles l'élément matche

		"""
		positions = []
		mp = self.mots_pleins(lexique)
		# print("longueur de mp :{} ; longueur de phono : {}".format(len(mp),len(self.phono)))

		# Pour chaque mot:
		for x in range(len(mp)):

			# si c'est un mot plein
			if mp[x] == 1:
				
				# Vérification que le token découpé en syllabes structurées contient le bon nombre de syllabes
				token = self.phono[x]
				syllabes = Contrepet.syllabation(token)

				# Si oui:
				if len(syllabes) == element.nb_syllabes: #>= avant

					position = [] # n° mot plein ; n° syllabe ; nature

					s = syllabes[element.syllabe-1]
					s = Contrepet.structure_syllabe(s)
					# print(s)
					# l'élément requis est une attaque consonnantique:
					if element.nature == "AC":
						if s["a"] and len(s["a"][0]) == 2:
							position = [x,element.syllabe-1,"a",0,2]
						else:
							continue
							
					# l'élément requis est un noyau:
					elif element.nature == "N":
						if syllabes[element.syllabe]["n"]:
							position = [x,element.syllabe-1,"n",0,2]


					# l'élément requis est une coda:
					elif element.nature == "C":
						if syllabes[element.syllabe]["c"]:
							position = [x,element.syllabe-1,"c",0,2]

					# l'élément requis est une attaque consonnantique complexe (cluster):	
					elif element.nature == "ACC":
						# print(s["a"],len(s["a"]))
						if s["a"] and len(s["a"][0]) == 4:
							position = [x,element.syllabe-1,"a",0,4]
						else:
							continue

					# l'élément requis est la première consonne de l'attaque consonnantique complexe (cluster):
					elif element.nature == "AC1":
						if s["a"] and len(s["a"][0]) == 4:
							position = [x,element.syllabe-1,"a",0,2]
						else:
							continue

					if len(position) > 0:
						positions.append(position)

				else:
					continue
			else:
				continue

		return positions



	def check(phono_token,lexique,ortho=False):
		"""
			Vérifier qu'une transcription phonétique correspond bien à un mot.
			- prend en entrée : une transcription phono., un lexique (phono -> ortho)
			- renvoit en sortie : l'orthographe si le mot existe, 'False' sinon.

		"""
		if phono_token in lexique.keys():
			if ortho == False:
				return False 
			elif ortho == True:
				return lexique[phono_token]
		else:
			if ortho == False:
				return False
			elif ortho == True:
				return None


	def switch(self,pos1,pos2,lexique):
		"""
			Echanger les segments phonétiques correspondant à deux positions.
			- prend en entrée : un objet Contrepet, deux positions et un lexique (ortho -> phono)
			- renvoit en sortie : la transcription phono. après permutation.

		"""
		def reconstruire_phono(str_syll_modif):
			""" 
				Reconstruire la transcription phono. d'une syllabe. 
				- prend en entrée : une structure syllabique
				- renvoie en sortie : les phonèmes concaténés (string).

			"""
			res = ""
			for s in str_syll_modif:
				l = ["a","n","c"]
				for elt in l:
					if s[elt]:
						res += str(s[elt][0])
					else:
						continue
			return res

		phono1 = self.get_syllabes(lexique)[pos1[0]][pos1[1]][pos1[2]]
		s1 = slice(pos1[3],pos1[4])
		phono1 = phono1[0][s1]

		phono2 = self.get_syllabes(lexique)[pos2[0]][pos2[1]][pos2[2]]
		s2 = slice(pos2[3],pos2[4])
		phono2 = phono2[0][s2]
		print("On va échanger {} avec {}".format(phono1,phono2))

		# si les deux phones sont différents (sinon pas de permutation)
		if phono1 != phono2:
			copie = self.get_syllabes(lexique)
			# on remplace phono1 par phono2
			phones = copie[pos1[0]][pos1[1]][pos1[2]][0]
			phones_p2 = phones[pos1[4]:]
			copie[pos1[0]][pos1[1]][pos1[2]] = [phono2 + phones_p2]

			# on remplace phono2 par phono1
			phones = copie[pos2[0]][pos2[1]][pos2[2]][0]
			phones_p2 = phones[pos2[4]:]
			copie[pos2[0]][pos2[1]][pos2[2]] = [phono1 + phones_p2]

			a = (pos1[0],reconstruire_phono(copie[pos1[0]]))
			b = (pos2[0],reconstruire_phono(copie[pos2[0]]))
			return a,b
		else:
			return None


	def appliquer_patron(self,lexique,patron):
		"""
		Appliquer un patron de permutations à une phrase d'input.
		- prend en entrée : un objet Contrepet, un lexique (ortho -> phono), un patron de permut.
		- renvoit en sortie : l'ensemble des candidats (transcriptions phonos.) pour cette phrase
			et ce patron.

		"""
		phono = self.phono
		echanges = []
		candidats = []

		# récupérer les possibilités d'échange pour chaque permutation
		for permutation in patron:
			echanges_permut = []
			positions1 = self.match(permutation[0],lexique)
			positions2 = self.match(permutation[1],lexique)
			# print(positions1,positions2)
			for p1 in positions1:
				for p2 in positions2:
					sp1sp2 = self.switch(p1,p2,lexique)
					print(sp1sp2)
					if sp1sp2 is not None:
						sp1sp2_tri = sorted(sp1sp2, key=lambda tup: tup[0])
						if sp1sp2_tri not in echanges_permut:
							echanges_permut.append(sp1sp2_tri)
			if len(echanges_permut) == 1:
				echanges = echanges_permut
			else:
				echanges.append(echanges_permut)
		toutes_poss = []
		if len(echanges) == 1:
			echanges = echanges[0]
			copie = list(phono)
			for echange in echanges:
				if type(echange) == tuple:
					copie[echange[0]] = echange[1]
					candidats.append(copie)
				else:
					copie = list(phono)
					for e in echange:
						copie[e[0]] = e[1]
						candidats.append(copie)
			return candidats
		else:
			toutes_poss = list(itertools.product(*echanges))
		for poss in toutes_poss:
			if poss[0] != poss[1]:
				tmp = poss[0] + poss[1]
				copie = list(phono)
				for e in tmp:
					copie[e[0]] = e[1]
				candidats.append(copie)
		uniq = []
		for candidat in candidats:
			if candidat not in uniq:
				uniq.append(candidat)
		if len(uniq) > 0:
			return uniq
		else: 
			return None


	def to_ortho(self,candidat_phono,lexique):
		"""
			Transcrire en orthographe une phrase transcrite en phonétique.
			- prend en entrée : un objet Contrepet, une transcription phonétique et un lexique (phono -> ortho)
			- renvoit en sortie : l'ensemble des orthographes possible pour une suite phono.

		"""
		candidats_ortho = []
		# candidat_ortho = []*len(self.ortho)
		# candidat_ortho = self.ortho.split()
		co = []
		for token in self.ortho:
			if "'" in token :
				token = token.split("'")
				co.extend(token)
			else:
				co.append(token)
		# print("co",co)

		self.ortho = list(co)
		candidat_ortho = self.ortho
		# print("candidat ortho : ",candidat_ortho)
		# print("self phono : ",self.phono)
		# print("candidat phono : ",candidat_phono)
		# tokens identiques:
		# for token in candidat_phono:
			# n = candidat_phono.index(token)
			# si ce token n'a pas subi de permuation:
			# if token == self.phono[n]:
				# print("ok pareil : ",token, self.phono[n])
				# on récupère le token ortho correspondant dans la V.O.
				# candidat_ortho[n] = self.ortho[n]
		# print("candidat ortho : ",candidat_ortho)
		# tokens différents :
		variantes = []
		var_pos = []
		# for token in candidat_phono:
		for n in range(len(self.phono)):
			# n = self.phono.index(token)
			token = candidat_phono[n]
			# si le token a subi une / des permutation(s)):
			if token != self.phono[n]:
				var_pos.append(n)
				# print("token différent", token,n)
				# on récupère les orthographes possibles pour ce token
				if Contrepet.check(token,lexique,ortho=True) is not None:
					ortho_poss = Contrepet.check(token,lexique,ortho=True)+[n]
					variantes.append(ortho_poss)
				else:
					return None
		# print("variantes",variantes)
		positions, combinaisons = combine(variantes)
		toutes_orthos = []
		for poss in combinaisons:
			tmp = list(candidat_ortho)
			# print("tmp",tmp)
			for token in poss:
				position = positions[poss.index(token)]
				# print("token, position",token,position)
				tmp[position] = token
				# print(len(tmp), tmp[3],tmp[12])
			toutes_orthos.append(tmp)
		return toutes_orthos



